/**********************************************
*  Window - David Chapman
*
*  This file opens a window for videogames.....
*  It sets up the opengl, keyboard, and sound
*  in the simplest way good enough for a FPS
*  game
**********************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef _WIN32
        #include <windows.h>
	#include <GL/gl.h>
	#include <GL/glu.h>
#endif
#include <GL/glut.h> 

#include "window.h"

/*global objects*/
Window window;

void InitOpenGL() {
	/*procedural opengl without lighting*/
	glEnable(GL_TEXTURE_2D);
	glShadeModel(GL_SMOOTH);
	glClearColor(0.0f, 0.0f, 0.0f, 0.5f);
	glClearDepth(1.0f);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
}
void Resize(int sizex, int sizey) {
	/*tell the rest of the stuff the window size*/
	window.x = sizex;
	window.y = sizey;

	/* Prevent A Divide By Zero */
	if (sizey==0)
		sizey=1;							

	/* Reset The Current Viewport */
	glViewport(0, 0, window.x, window.y);

	/* Calculate The FOV of The Window */
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(45.0f,(GLfloat)sizex/(GLfloat)sizey,0.1f,20000.0f);

	/* Make the FOV right hand rule
	   X is right, Y is forward, Z is up*/
	float rightHandIdentity[] = {
		1, 0, 0, 0,
		0, 0, 1, 0,
		0,-1, 0, 0,
		0, 0, 0, 1};
	glMultTransposeMatrixf(rightHandIdentity);
}
void SpecKeyDown(int key, int x, int y) {
	if (key < 256) window.speckeys[key] = 1;
}
void SpecKeyUp(int key, int x, int y) {
	if (key < 256) window.speckeys[key] = 0;
}
void KeyDown(unsigned char key, int x, int y) {
	window.keys[key] = 1;
}
void KeyUp(unsigned char key, int x, int y) {
	window.keys[key] = 0;
}
void MouseClick(int button, int state, int x, int y) {
	window.click[button] = state;
}
void (*WindowDraw)() = NULL;
void (*WindowKey)() = NULL;
void (*WindowUpdate)() = NULL;
void (*WindowMouse)(int x, int y) = NULL;
void MouseMove(int x, int y) {
	/*figure out the pos relative to the window center*/
	x -= window.x/2;
	y -= window.y/2;

	/*rotate the camera*/
	WindowMouse(x, y);
}
void Idle() {
	/*compute the fps*/
	float prevtime = window.time;
	window.time = glutGet(GLUT_ELAPSED_TIME)*0.001f;
	window.spf = window.time - prevtime;

	/*reset the mouse to the center of the window*/
	glutWarpPointer(window.x/2,window.y/2);

	WindowKey();
	WindowUpdate();
	WindowDraw();
}

int WindowCreate(
int argc, char **argv,
void (*initfunc)(),
void (*updatefunc)(), 
void (*drawfunc)(), 
void (*keyfunc)(), 
void (*mousefunc)(int x, int y),
int width, int height, int fullscreen
) {

	/*set the callbacks*/
	WindowUpdate = updatefunc;
	WindowDraw   = drawfunc;
	WindowKey    = keyfunc;
	WindowMouse  = mousefunc;

	/*supposedly the window starts at 60fps*/
	window.time = 0.0f;
	window.spf = 1.0f/60.0f;

	/*the camera is initially not doing anything....
	  all keys initially unpressed*/
	memset(window.keys, 0, 256);
	memset(window.speckeys, 0, 256);

	/*make and configure the window*/
	glutInit(&argc, argv);
	glutInitDisplayMode(GLUT_DEPTH | GLUT_DOUBLE | GLUT_RGBA);
	glutInitWindowPosition(0,0);
	glutInitWindowSize(width,height);
	glutCreateWindow("David Chapman's Netcode Tutorial");
	if (fullscreen)
		glutFullScreen();
	glutReshapeFunc(Resize);
	glutDisplayFunc(Idle);
	glutIdleFunc(Idle);
	glutKeyboardFunc  (KeyDown);
	glutKeyboardUpFunc(KeyUp);
	glutSpecialFunc   (SpecKeyDown);
	glutSpecialUpFunc (SpecKeyUp);
	glutMotionFunc(MouseMove);
	glutPassiveMotionFunc(MouseMove); 
	glutSetCursor(GLUT_CURSOR_NONE);
	InitOpenGL();
	initfunc();	/*init code for the game!*/

	/*run the loop*/
	glutMainLoop();

	return 0;
}


