/***************************
* This is a heavily modified *butchered* version
* of a tutorial by Gillaume Cottenceau
*
* It allows the engine to read/write pngs!
****************************/
/*
 * Copyright 2002-2008 Guillaume Cottenceau.
 *
 * This software may be freely redistributed under the terms
 * of the X11 license.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#define PNG_DEBUG 3
#include <png.h>

#include "image.h"

void abort_(const char * s, ...)
{
	va_list args;
	va_start(args, s);
	vfprintf(stderr, s, args);
	fprintf(stderr, "\n");
	va_end(args);
	exit(-5);
}


Image ImageReadPng(const char* filename) {

	/*png data*/
	int x, y, i, j;
	int width, height;
	png_byte color_type;
	png_byte bit_depth;
	png_structp png_ptr;
	png_infop info_ptr;
	int number_of_passes;
	png_bytep * row_pointers;
	Image image;

	char header[8];	/* 8 is the maximum size that can be checked */

	/* open file and test for it being a png */
	FILE *fp = fopen(filename, "rb");
	if (!fp)
		abort_("[read_png_file] File %s could not be opened for reading", filename);
	int rt=fread(header, 1, 8, fp);
	if (png_sig_cmp((png_bytep)header, 0, 8))
		abort_("[read_png_file] File %s is not recognized as a PNG file", filename);


	/* initialize stuff */
	png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	
	if (!png_ptr)
		abort_("[read_png_file] png_create_read_struct failed");

	info_ptr = png_create_info_struct(png_ptr);
	if (!info_ptr)
		abort_("[read_png_file] png_create_info_struct failed");

	if (setjmp(png_jmpbuf(png_ptr)))
		abort_("[read_png_file] Error during init_io");

	png_init_io(png_ptr, fp);
	png_set_sig_bytes(png_ptr, 8);

	png_read_info(png_ptr, info_ptr);

	width = info_ptr->width;
	height = info_ptr->height;
	color_type = info_ptr->color_type;
	bit_depth = info_ptr->bit_depth;

	number_of_passes = png_set_interlace_handling(png_ptr);
	png_read_update_info(png_ptr, info_ptr);


	/* read file */
	if (setjmp(png_jmpbuf(png_ptr)))
		abort_("[read_png_file] Error during read_image");

	row_pointers = (png_bytep*) malloc(sizeof(png_bytep) * height);
	for (y=0; y<height; y++)
		row_pointers[y] = (png_byte*) malloc(info_ptr->rowbytes);

	png_read_image(png_ptr, row_pointers);

        fclose(fp);

	/*initialize the engine format image*/
	image.sizex = width;
	image.sizey = height;
	image.data = malloc(width*height*4);
	
	/*copy the data line by line*/
	unsigned char *out = image.data;
	for (i = 0; i < image.sizey; i++) {
		/*if it's rgba, copy it straight! */
		if (info_ptr->color_type == PNG_COLOR_TYPE_RGBA) {
			memcpy(out, row_pointers[i], image.sizex*4);
			out += image.sizex*4;
		}

		/*otherwise copy 3, and invent a fourth byte*/
		else if (info_ptr->color_type == PNG_COLOR_TYPE_RGB) {
			for (j = 0; j < image.sizex; j++) {
				out[0] = row_pointers[i][3*j + 0];
				out[1] = row_pointers[i][3*j + 1];
				out[2] = row_pointers[i][3*j + 2];
				out[3] = 255;
				out += 4;
			}
		}
		else
			abort_("PNG %s must be RGB or RGBA", filename);
	}

        /* cleanup heap allocation */
	for (y=0; y<height; y++)
		free(row_pointers[y]);
	free(row_pointers);

	return image;
}

void ImageWritePng(Image *image, const char* filename) {

	int x, y;
	int width = image->sizex;
	int height = image->sizey;
	png_byte color_type = PNG_COLOR_TYPE_RGBA;
	png_byte bit_depth = 8;
	png_structp png_ptr;
	png_infop info_ptr;
	png_bytep * row_pointers;

	/*allocate some row pointers, and make them point
	  into the image data*/
	row_pointers = malloc(height*sizeof(png_bytep*));
	for (y = 0; y < height; y++)
		row_pointers[y] = image->data + y*width*4;

	/* create file */
	FILE *fp = fopen(filename, "wb");
	if (!fp)
		abort_("[write_png_file] File %s could not be opened for writing", filename);


	/* initialize stuff */
	png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	
	if (!png_ptr)
		abort_("[write_png_file] png_create_write_struct failed");

	info_ptr = png_create_info_struct(png_ptr);
	if (!info_ptr)
		abort_("[write_png_file] png_create_info_struct failed");

	if (setjmp(png_jmpbuf(png_ptr)))
		abort_("[write_png_file] Error during init_io");

	png_init_io(png_ptr, fp);


	/* write header */
	if (setjmp(png_jmpbuf(png_ptr)))
		abort_("[write_png_file] Error during writing header");

	png_set_IHDR(png_ptr, info_ptr, width, height,
		     bit_depth, color_type, PNG_INTERLACE_NONE,
		     PNG_COMPRESSION_TYPE_BASE, PNG_FILTER_TYPE_BASE);

	png_write_info(png_ptr, info_ptr);


	/* write bytes */
	if (setjmp(png_jmpbuf(png_ptr)))
		abort_("[write_png_file] Error during writing bytes");

	png_write_image(png_ptr, row_pointers);


	/* end write */
	if (setjmp(png_jmpbuf(png_ptr)))
		abort_("[write_png_file] Error during end of write");

	png_write_end(png_ptr, NULL);

	free(row_pointers);

        fclose(fp);
}
