#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "SDL/SDL.h"
#include "SDL/SDL_mixer.h"

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>

#include "image.h"
#include "gamestate.h"
#include "window.h"
#include "tcp.h"
#include "buffered_tcp.h"

#define TRUE 1
#define FALSE 0
#define SCREEN_X 800
#define SCREEN_Y 600
#define PI 3.14159265358979323846264
#define MEGABYTE (1024*1024)

//----------------------
// GameState objects
//
// Gamestate is a godlike struct that contains
//  the positions and current state of every object
//  in the game.  It's already "serialized",
//  so we can just send it across the network to
//  another machine to tell it what's going on.
//
//----------------------
GameState gameState;
int playerID;
int numPlayers;

//----------------------
//
// By the time we receive the "serverState", 
//  it's already out of date.  Plus, we don't
//  want the server to move our ship.
//
// We combine serverState, along with our own player
//  positions and keyboard motion, to produce "gamestate"
//  that we shown on the screen.
//
// In the end "gamestate" is only a best-guess of what's really
//  going on ..... If your ping is low enough, then
//  "gamestate" is accurate enough to enjoy the game.
//
//----------------------
TcpBuffer *sock;
GameState serverState;
void SendPlayerState();
void RecvServerState();
void AssimilateServerState();

//----------------------
// Rendering Functions
//----------------------
int asteroid_tex;
int blast_tex;
int explode_tex;
int ship_tex[2];
void DrawWrapSprite(int tex, float x, float y, float sizex, float sizey, float rot);
void DrawSprite(int tex, float x, float y, float sizex, float sizey, float rot);
int LoadTextureFromPng(const char *filename);


//----------------------
// Sound stuff
//----------------------
void LoadSounds();
SDL_Surface *screen;			//Pointer to the main screen surface
Mix_Chunk *sound_blast = NULL;		//Pointer to our sound, in memory
Mix_Chunk *sound_explode = NULL;	//Pointer to our sound, in memory
int channel;				//Channel on which our sound is played
  
int audio_rate = 22050;			//Frequency of audio playback
Uint16 audio_format = AUDIO_S16SYS; 	//Format of the audio we're playing
int audio_channels = 2;			//2 channels = stereo
int audio_buffers = 4096;		//Size of the audio buffers in memory


//------------------------------------
//------------------------------------
// Main Game functions  (ordered by importantce)
//------------------------------------
//------------------------------------

// Called every frame to update objects
void Update() {

	// Run the simulation!
	UpdateGameState(&gameState, window.spf);
	SendPlayerState();
	RecvServerState();

	// Play the sounds
	int s;
	for (s=0; s<gameState.numSounds; s++) {
		if (gameState.sounds[s].type == SOUND_BLAST)
			Mix_PlayChannel(-1, sound_blast, 0);
		else
			Mix_PlayChannel(-1, sound_explode, 0);
	}
	gameState.numSounds = 0;
}

// Called at start of program
void LoadData() {

printf("LD 10\n");

	// Load the textures
	asteroid_tex = LoadTextureFromPng("data/asteroid.png");
	blast_tex    = LoadTextureFromPng("data/blast.png");
	explode_tex  = LoadTextureFromPng("data/explode.png");
	ship_tex[0]  = LoadTextureFromPng("data/ship2.png");
	ship_tex[1]  = LoadTextureFromPng("data/ship.png");

printf("LD 20\n");
	// Load the sounds
	LoadSounds();

printf("LD 30\n");
	// Initialize the new level
	BufferedTcpRecv(sock, &numPlayers, sizeof(int));
	BufferedTcpRecv(sock, &playerID, sizeof(int));

printf("LD 40\n");
	NewGameState(&gameState, 10, numPlayers);
	playerID = 0;
}

void LoadSounds() {
	if (SDL_Init(SDL_INIT_AUDIO) != 0) {
		printf("Unable to initialize SDL: %s\n", SDL_GetError());
		exit(1);
	}
	
	//Initialize SDL_mixer with our chosen audio settings
	if(Mix_OpenAudio(audio_rate, audio_format, audio_channels, audio_buffers) != 0) {
		printf("Unable to initialize audio: %s\n", Mix_GetError());
		exit(1);
	}
	
	//Load our WAV file from disk
	sound_blast = Mix_LoadWAV("data/blast.wav");
	if(sound_blast == NULL) {
		printf("Unable to load WAV file: %s\n", Mix_GetError());
	}
	sound_explode = Mix_LoadWAV("data/explosion.wav");
	if(sound_explode == NULL) {
		printf("Unable to load WAV file: %s\n", Mix_GetError());
	}

	//Play our sound file, and capture the channel on which it is played
	//channel = Mix_PlayChannel(-1, sound, 0);
	//if(channel == -1) {
	//	printf("Unable to play WAV file: %s\n", Mix_GetError());
	//}
}

// Called every frame to draw objects
void Draw(void) {

	int p,a,b,e;

	//clear the scene
	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glColor4f(1.0,1.0,1.0,1.0);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_BLEND);

	//2D camera perspective
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity ();                          // Reset The Projection Matrix
	gluOrtho2D(0, SCREEN_X, 0, SCREEN_Y);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	// Draw the blasts
	for (p=0; p<gameState.numPlayers; p++) {
		for (b=0; b<gameState.players[p].numBlasts; b++) {
			DrawWrapSprite(blast_tex,
				gameState.players[p].blasts[b].pos.x,
				gameState.players[p].blasts[b].pos.y,
				32, 32, 0.0);
		}
	}

	// Draw the ships
	for (p=0; p<gameState.numPlayers; p++) {
		DrawWrapSprite(ship_tex[p],
			gameState.players[p].ship.pos.x,
			gameState.players[p].ship.pos.y,
			SHIP_SIZE, SHIP_SIZE,
			gameState.players[p].ship.rot);
	}

	// Draw the asteroids
	for (a=0; a<gameState.numAsteroids; a++) {

		// How big is it?
		int size=0;
		switch (gameState.asteroids[a].type) {
			case ASTEROID_TYPE_SMALL:
				size = ASTEROID_SIZE_SMALL;
			break;
			case ASTEROID_TYPE_MEDIUM:
				size = ASTEROID_SIZE_MEDIUM;
			break;
			case ASTEROID_TYPE_LARGE:
				size = ASTEROID_SIZE_LARGE;
			break;
			default:
				printf("Asteroid with unknown size\n");
				exit (1);
			break;
		}

		// Draw it !
		DrawWrapSprite(asteroid_tex,
			gameState.asteroids[a].pos.x,
			gameState.asteroids[a].pos.y,
			size, size,
			0);
	}


	// Draw the explosions
	glBlendFunc(GL_SRC_COLOR, GL_ONE_MINUS_SRC_COLOR);
	for (e=0; e<gameState.numExplosions; e++) {
		float delta = (float)gameState.explosions[e].age / (float)EXPLOSION_DURATION;
		float scale   = 40+80*delta;
		float opacity = 1.0 - delta;
		glColor4f(opacity,opacity,opacity,opacity);
		DrawWrapSprite(explode_tex, 
			gameState.explosions[e].pos.x, 
			gameState.explosions[e].pos.y, 
			scale, scale, 0.0);
	}
	glColor4f(1.0,1.0,1.0,1.0);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	//double buffering
	glFlush();
	glutSwapBuffers();
}

//------------------------------------------------------
//------------------------------------------------------
// Rendering Functions
//------------------------------------------------------
//------------------------------------------------------
void DrawWrapSprite(int tex, float x, float y, float sizex, float sizey, float rot) {
	// Draw a bunch of copies because the thing wraps around the world
	DrawSprite(tex, x, y, sizex, sizey, rot);
	DrawSprite(tex, x+SCREEN_X, y,          sizex, sizey, rot);
	DrawSprite(tex, x-SCREEN_X, y,          sizex, sizey, rot);
	DrawSprite(tex, x,          y+SCREEN_Y, sizex, sizey, rot);
	DrawSprite(tex, x,          y-SCREEN_Y, sizex, sizey, rot);
}
void DrawSprite(int tex, float x, float y, float sizex, float sizey, float rot) {

	glBindTexture(GL_TEXTURE_2D, tex);
	glPushMatrix();
	glTranslatef(x,y,0.0);
	glRotatef(rot, 0.0,0.0,1.0);
	glScalef(0.5*sizex, 0.5*sizey, 1.0);

	//Draw the Box
	glBegin(GL_QUADS);
		glTexCoord2f(0.0,0.0);   glVertex3f(-1.0,-1.0,0.0);
		glTexCoord2f(1.0,0.0);   glVertex3f( 1.0,-1.0,0.0);
		glTexCoord2f(1.0,1.0);   glVertex3f( 1.0, 1.0,0.0);
		glTexCoord2f(0.0,1.0);   glVertex3f(-1.0, 1.0,0.0);
	glEnd();

	glPopMatrix();
}
int LoadTextureFromPng(const char *filename) {

	unsigned int texID;

	// Read the Png
	Image img = ImageReadPng(filename);

	// Load the texture
	glGenTextures(1, &texID);
	glBindTexture(GL_TEXTURE_2D, texID);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
	glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
	glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
	gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGBA, img.sizex, img.sizey, GL_RGBA, GL_UNSIGNED_BYTE, img.data);

	// no leaks
	free(img.data);

	return texID;
}


//------------------------------------
//------------------------------------
// TCP packet functions
//------------------------------------
//------------------------------------
void SendPlayerState() {

	// Send at 50 fps.... so abort if it
	//  hasn't been 1/50 seconds since last send
	static double lastTcpSendTime=0.0;
	if ((window.time-lastTcpSendTime) < (1.0/50.0))
		return;

	// The sent client's ship to the server
	Ship *ship = &(gameState.players[playerID].ship);
	BufferedTcpSend(sock,ship,sizeof(Ship));
	lastTcpSendTime = window.time;

	// The ship is no longer firing
	ship->isFiring = FALSE;
}
void RecvServerState() {

	// The server may have queued up a bunch
	//  of gamestates in our "mailbox" (tcp stack)
	//    (although it probably only sent us one!)
	//  read all of them
	while (  BufferedTcpPeekRecv(sock, sizeof(GameState)) ) {
		BufferedTcpRecv(sock,&serverState,sizeof(GameState));
		AssimilateServerState();
	}
}
void AssimilateServerState() {

	int i;

	//-------------------------
	// Who do we trust?  Me or the Server?
	//
	// For everything except my ship
	//  I trust the server
	//-------------------------

	// backup my ship
	Ship myShip = gameState.players[ playerID ].ship;

	// copy over the server's state
	gameState = serverState;

	// restore my ship
	gameState.players[ playerID ].ship = myShip;

}


void Key() {
	int i;

	// if the ship isn't stunned
	if (gameState.players[playerID].stunDuration <= 0.0) {

		//handle every key pressed.....
		if (window.speckeys[ GLUT_KEY_LEFT ]) {
			ShipRot(&gameState, playerID, 1.0, window.spf);
		}
		if (window.speckeys[ GLUT_KEY_RIGHT ]) {
			ShipRot(&gameState, playerID,-1.0, window.spf);
		}
		if (window.speckeys[ GLUT_KEY_UP ]) {
			ShipAccel(&gameState, playerID, window.spf);
		}
		if (window.speckeys[ GLUT_KEY_DOWN ]) {
			ShipDecel(&gameState, playerID, window.spf);
		}

		if (window.keys[ ' ' ]) {
			ShipBlast(&gameState, playerID);
		}
	}
}
void Mouse(int x, int y) {
}

int main(int argc, char **argv) {

	// Read the command arguments
	if (argc!=3) {
		printf("USAGE:\n");
		printf("   ./aster_client ip port\n");
		printf("\n");
		printf("     for example:  ./aster_client localhost 7777\n");
		printf("      will join a a game with deticated server with IP address 'localhost' and port 7777\n");
		exit(1);
	}
	char *ip   =      argv[1];
	int   port = atoi(argv[2]);
printf("main 10\n");
	// Connect to the deticated server
	sock = BufferedTcpClientConnect(ip, port, 1, 0, MEGABYTE);

printf("main 20\n");
	// Make the glut window
	WindowCreate(
		argc, argv,
		LoadData,
		Update,
		Draw,
		Key,
		Mouse,
	SCREEN_X,SCREEN_Y,0);

	return 0;
}


