// by David Chapman for UMBC Game Developers Club

#ifndef _GAMESTATE_H_
#define _GAMESTATE_H_

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

//---------------------
// Global Game Constants
//---------------------
#define WORLD_X 800
#define WORLD_Y 600
#define ASTEROID_SIZE_SMALL  20
#define ASTEROID_SIZE_MEDIUM 40
#define ASTEROID_SIZE_LARGE  80
#define ASTEROID_SPEED 10.0
#define ASTEROID_NUM_SHARDS 3
#define BLAST_SPEED 1000.0
#define BLAST_DURATION 1.0
#define BLAST_COOLDOWN 0.5
#define SHIP_ROTATE 200.0
#define SHIP_ACCEL 100.0
#define SHIP_SIZE  60
#define SHIP_STUN_DURATION 2
#define EXPLOSION_DURATION 0.5
#define PLAYER_START_OFFSET_X 80

//---------------------
// a 2D x,y coordinate
//---------------------
typedef struct Coord {
	float x, y;
} Coord;
Coord CoordSet(float x, float y);
void MoveCoord(Coord *pos, Coord offset, float spf);
void LoopAroundWorld(Coord *coord);

//---------------------
// The Asteroids game-objects
//---------------------
#define ASTEROID_TYPE_SMALL  1
#define ASTEROID_TYPE_MEDIUM 2
#define ASTEROID_TYPE_LARGE  3
typedef struct Asteroid {
	Coord pos;
	Coord speed;
	int type;
} Asteroid;
typedef struct Ship {
	Coord pos;
	Coord speed;
	Coord accel;
	float blastCooldown;
	float rot;   // in degrees
	int isFiring;
} Ship;
typedef struct Explosion {
	Coord pos;
	float age;
} Explosion;
typedef struct Blast {
	Coord pos;
	Coord speed;
	float age;
} Blast;
#define SOUND_BLAST   1
#define SOUND_EXPLODE 2
typedef struct Sound {
	int type;
} Sound;

//------------------------------------------
//------------------------------------------
//  :) :) :) :)  GameState  <3 <3 <3 <3 <3
//                Object
//
// 1.  This object is completely serialized (NO POINTERS  NO VECTORS  NO LISTS)
// 2.  This object is of fixed length (FIXED SIZE ARRAYS)
//
//  Therefore, by 1. and 2.  we can just
//    send the entire GameState object over TCP (NetCode!)
//
//------------------------------------------
//------------------------------------------
#define MAX_ASTEROIDS 100
#define MAX_PLAYERS 2
#define MAX_BLASTS 100
#define MAX_SOUNDS 10
#define MAX_EXPLOSIONS 100
typedef struct PlayerState {
	Ship ship;
	Blast blasts[MAX_BLASTS];
	int numBlasts;

	float stunDuration;
} PlayerState;
typedef struct GameState {

	// What are the game objects
	PlayerState  players   [MAX_PLAYERS];
	Asteroid     asteroids [MAX_ASTEROIDS];
	Explosion    explosions[MAX_EXPLOSIONS];
	Sound        sounds    [MAX_SOUNDS];

	// how many of each?
	int numPlayers;
	int numAsteroids;
	int numExplosions;
	int numSounds;
} GameState;


//---------------------------------------
// This function is called at the start of 
//  the game, to set up a new level
//---------------------------------------
void NewGameState(GameState *gs, int numAsteroids, int numPlayers);

//---------------------------------------
// This function is called every frame
//  to move all of the objects around
//  in the world
//---------------------------------------
void UpdateGameState(GameState *gs, float spf);
void ArbitrateGameState(GameState *gs, float spf);


//---------------------------------------
// These functions are called when we press keyboard buttons
//---------------------------------------
void ShipAccel(GameState *gs, int player, float spf);
void ShipDecel(GameState *gs, int player, float spf);
void ShipRot(GameState *gs, int player, float scale, float spf);
void ShipBlast(GameState *gs, int player);



// Add/Remove from Gamestate Object
void AddAsteroid( GameState *gs, Coord pos, Coord speed, int type );
void AddExplosion( GameState *gs, Coord pos );
void AddBlast( GameState *gs, int player, Coord pos, Coord speed );
void AddSound( GameState *gs, int type );
Asteroid DeleteAsteroid(GameState *gs, int idx);
Explosion DeleteExplosion(GameState *gs, int idx);
Blast DeleteBlast(GameState *gs, int player, int idx);


float RandF();
Coord RandomDirection(float speed);
Coord RandomPosition(float minX, float maxX, float minY, float maxY);



#endif // _GAMESTATE_H_

