#include "gamestate.h"

#define TRUE 1
#define FALSE 0
#define PI 3.14159265358979323846264

// Local prototypes
int BlastHitAsteroids(GameState *gs, Blast blast);
int IsBlastHitAsteroid(Blast blast, Asteroid asteroid);
int IsShipHitAsteroid(Ship ship, Asteroid asteroid);
Asteroid AsteroidSplitShards(GameState *gs, int asteroidIdx);

//---------------------------------------
//---------------------------------------
// This function is called at the start of 
//  the game, to set up a new level
//---------------------------------------
//---------------------------------------
void NewGameState(GameState *gs, int numAsteroids, int numPlayers)
{
	int p, a;

	// Initialize the gamestate to nothing
	gs->numPlayers = 0;
	gs->numAsteroids = 0;
	gs->numExplosions = 0;
	gs->numSounds = 0;

	// X Players
	gs->numPlayers = numPlayers;
	for (p=0; p<numPlayers; p++) {

		gs->players[p].stunDuration = 0.0;

		// no blasts
		gs->players[p].numBlasts = 0;

		// Ship in the middle
		gs->players[p].ship.pos   = CoordSet( WORLD_X/2.0,  WORLD_Y/2.0 );
		gs->players[p].ship.speed = CoordSet( 0.0, 0.0 );
		gs->players[p].ship.accel = CoordSet( 0.0, 0.0 );
		gs->players[p].ship.blastCooldown = 0.0;
		gs->players[p].ship.rot = 90.0;   // pointed up
		gs->players[p].ship.isFiring = FALSE;

		// Move the ship by the player offset
		gs->players[p].ship.pos.x += p * PLAYER_START_OFFSET_X;
	}

	// Make some asteroids
	for (a=0; a<numAsteroids; a++) {
		Coord randPos   = RandomPosition (0,WORLD_X, 0,WORLD_Y);
		Coord randSpeed = RandomDirection(ASTEROID_SPEED);
		AddAsteroid( gs, randPos, randSpeed, ASTEROID_TYPE_LARGE );
	}
}


//---------------------------------------
//---------------------------------------
// This function is called every frame
//  to move all of the objects around
//  in the world
//---------------------------------------
//---------------------------------------
void UpdateGameState(GameState *gs, float spf) {

	GameState new;
	int a, b, p, e, s;  // loopers

	//-------------------------
	// Explosions get older by one frame...
	//   (spf  is   seconds per frame)
	//-------------------------
	for (e=0; e<gs->numExplosions; e++) {
		gs->explosions[e].age += spf;  // age

		// if we're too old then remove
		if (gs->explosions[e].age > EXPLOSION_DURATION) {
			DeleteExplosion(gs, e);
			e--;   // fix the for loop iterator...(idx e replaced in delete)
		}
	}

	//-------------------------
	// Ships get older too (blast cooldown)
	//  players get un-stunned
	//  and so do the blasts...( they die! )
	//-------------------------
	for (p=0; p<gs->numPlayers; p++) {
		gs->players[p].ship.blastCooldown += spf;

		if (gs->players[p].stunDuration > 0.0)
			gs->players[p].stunDuration -= spf;

		for (b=0; b<gs->players[p].numBlasts; b++) {
			gs->players[p].blasts[b].age += spf;
			if (gs->players[p].blasts[b].age > BLAST_DURATION) {
				DeleteBlast(gs, p, b);
				b--;   // fix the for loop iterator...(idx e replaced in delete)
			}
		}
	}

	//-------------------------
	// PHYSICS !!  (or there abouts)
	// All of the objects are flying around
	//-------------------------
	// Move Asteroids
	for (a=0; a<gs->numAsteroids; a++) {
		MoveCoord(&gs->asteroids[a].pos, gs->asteroids[a].speed, spf);
		LoopAroundWorld(&gs->asteroids[a].pos);
	}
	for (p=0; p<gs->numPlayers; p++) {
		// Move Blasts
		for (b=0; b<gs->players[p].numBlasts; b++) {
			MoveCoord(&gs->players[p].blasts[b].pos, gs->players[p].blasts[b].speed, spf);
			LoopAroundWorld(&gs->players[p].blasts[b].pos);
		}
		if (gs->players[p].stunDuration <= 0.0) {
			// Move Ship
			MoveCoord(&gs->players[p].ship.pos,   gs->players[p].ship.speed, spf);
			MoveCoord(&gs->players[p].ship.speed, gs->players[p].ship.accel, spf);
			LoopAroundWorld(&gs->players[p].ship.pos);
		}
	}
}

void ArbitrateGameState(GameState *gs, float spf) {

	int p,b,a;

	//-------------------------
	// Make Blasts
	//-------------------------
	for (p=0; p<gs->numPlayers; p++) {
		if ( (gs->players[p].ship.isFiring) ) {
			// Where's the ship, and where's it pointing
			Coord pos = gs->players[ p ].ship.pos;
			Coord dir = CoordSet(
			               cos( (PI/180.0)*gs->players[ p ].ship.rot),
			               sin( (PI/180.0)*gs->players[ p ].ship.rot) );
			Coord speed = CoordSet( BLAST_SPEED*dir.x, BLAST_SPEED*dir.y );

			// Add the blast and the sound
			AddBlast( gs, p, pos, speed );
			AddSound( gs, SOUND_BLAST );

			// The ship is no longer firing
			gs->players[p].ship.isFiring = FALSE;
		}
	}

	//-------------------------
	// Collide Asteroid with Blast
	//-------------------------
	for (p=0; p<gs->numPlayers; p++) {
		for (b=0; b<gs->players[p].numBlasts; b++) {

			// If we hit an asteroid...
			if (BlastHitAsteroids(gs, gs->players[p].blasts[b]))
			{
				// Delete the Blast
				Blast blast = DeleteBlast(gs, p, b);

				// Then create an explosion
				AddExplosion(gs, blast.pos);

				// with a BOOM sound
				AddSound(gs, SOUND_EXPLODE);
			}
		}
	}

	//-------------------------
	// Collide Ship with Asteroid
	//-------------------------
	for (p=0; p<gs->numPlayers; p++) {
		for (a=0; a<gs->numAsteroids; a++) {
			if (IsShipHitAsteroid(gs->players[p].ship, gs->asteroids[a])) {
				Asteroid dead = AsteroidSplitShards(gs, a);
				a--;  // one fewer asteroid to iterate over

				// Then create an explosion
				AddExplosion(gs, dead.pos);

				// with a BOOM sound
				AddSound(gs, SOUND_EXPLODE);

				// Stun the ship
				gs->players[p].stunDuration = SHIP_STUN_DURATION;
			}
		}
	}
}
int BlastHitAsteroids(GameState *gs, Blast blast)
{
	int a;

	// Look if we hit anything
	int hitAnything = FALSE;
	int hitIdx = -1;
	for (a=0; a<gs->numAsteroids; a++) {
		if ( IsBlastHitAsteroid(blast, gs->asteroids[a]) ) {
			hitAnything = TRUE;
			hitIdx = a;
		}
	}

	// If we hit, then blow it up
	if (hitAnything) {
		AsteroidSplitShards(gs, hitIdx);
	}

	return hitAnything;
}
Asteroid AsteroidSplitShards(GameState *gs, int asteroidIdx) {
	int a;
	Asteroid dead = DeleteAsteroid(gs, asteroidIdx);

	// Add X smaller asteroids
	if (dead.type == ASTEROID_TYPE_LARGE)
		for (a=0; a<ASTEROID_NUM_SHARDS; a++)
			AddAsteroid( gs, dead.pos, RandomDirection(ASTEROID_SPEED), ASTEROID_TYPE_MEDIUM );
	if (dead.type == ASTEROID_TYPE_MEDIUM)
		for (a=0; a<ASTEROID_NUM_SHARDS; a++)
			AddAsteroid( gs, dead.pos, RandomDirection(ASTEROID_SPEED), ASTEROID_TYPE_SMALL );
	return dead;
}
int IsBlastHitAsteroid(Blast blast, Asteroid asteroid) {
	// the distance between them?
	float x = asteroid.pos.x - blast.pos.x;
	float y = asteroid.pos.y - blast.pos.y;
	float dist = sqrt( x*x + y*y );

	// Is it within the size of the asteroid?
	switch (asteroid.type) {
		case ASTEROID_TYPE_SMALL:
			return (dist < 0.5*ASTEROID_SIZE_SMALL);
		break;
		case ASTEROID_TYPE_MEDIUM:
			return (dist < 0.5*ASTEROID_SIZE_MEDIUM);
		break;
		case ASTEROID_TYPE_LARGE:
			return (dist < 0.5*ASTEROID_SIZE_LARGE);
		break;
	}
	// Uuuuh we should get here
	printf ("ERROR: unknown asteroid\n");
	exit(1);
	return FALSE;  // shut up compiler
}
int IsShipHitAsteroid(Ship ship, Asteroid asteroid) {
	// the distance between them?
	float x = asteroid.pos.x - ship.pos.x;
	float y = asteroid.pos.y - ship.pos.y;
	float dist = sqrt( x*x + y*y );

	// Is it within the size of the asteroid?
	switch (asteroid.type) {
		case ASTEROID_TYPE_SMALL:
			return (dist < 0.5*ASTEROID_SIZE_SMALL + 0.5*SHIP_SIZE);
		break;
		case ASTEROID_TYPE_MEDIUM:
			return (dist < 0.5*ASTEROID_SIZE_MEDIUM + 0.5*SHIP_SIZE);
		break;
		case ASTEROID_TYPE_LARGE:
			return (dist < 0.5*ASTEROID_SIZE_LARGE + 0.5*SHIP_SIZE);
		break;
	}
	// Uuuuh we should get here
	printf ("ERROR: unknown asteroid\n");
	exit(1);
	return FALSE;  // shut up compiler
}

//---------------------------------------
//---------------------------------------
// These functions are called when we press keyboard buttons
//---------------------------------------
//---------------------------------------
void ShipAccel(GameState *gs, int player, float spf) {

	// Get the direction of acceleration
	float theta = (PI/180.0) * gs->players[player].ship.rot;
	Coord accelDir = CoordSet( cos(theta), sin(theta) );

	// Accelerate it!
	MoveCoord( & gs->players[player].ship.speed, accelDir, spf*SHIP_ACCEL );
}
void ShipDecel(GameState *gs, int player, float spf) {

	// Get the direction of acceleration
	float theta = (PI/180.0) * gs->players[player].ship.rot;
	Coord accelDir = CoordSet( cos(theta), sin(theta) );

	// Accelerate it!
	MoveCoord( & gs->players[player].ship.speed, accelDir, -spf*SHIP_ACCEL );
}
void ShipRot(GameState *gs, int player, float scale, float spf) {
	gs->players[ player ].ship.rot += SHIP_ROTATE * scale * spf;
}
void ShipBlast(GameState *gs, int player) {

	if (gs->players[ player ].ship.blastCooldown > BLAST_COOLDOWN)
	{
		// Make the blast
		gs->players[ player ].ship.isFiring=TRUE;
		gs->players[ player ].ship.blastCooldown=0.0;
	}
}

//------------------------------------
// Objects moving around the world
//------------------------------------
Coord CoordSet(float x, float y) {
	Coord c;
	c.x = x;
	c.y = y;
	return c;
}
void MoveCoord(Coord *pos, Coord offset, float spf) {
	pos->x += spf * offset.x;
	pos->y += spf * offset.y;
}
void LoopAroundWorld(Coord *coord) {
	// A "Toroidal" coordinate system
	while (coord->x < 0)
		coord->x += WORLD_X;
	while (coord->x >= WORLD_X)
		coord->x -= WORLD_X;
	while (coord->y < 0)
		coord->y += WORLD_Y;
	while (coord->y >= WORLD_Y)
		coord->y -= WORLD_Y;
}

//------------------------------------
// Add new game objects
//   (Copy + Paste anyone?)
//------------------------------------
void AddAsteroid( GameState *gs, Coord pos, Coord speed, int type )
{
	// Make an asteroid
	Asteroid a;
	a.pos = pos;
	a.speed = speed;
	a.type = type;

	// Add to the end of the list
	if (gs->numAsteroids < MAX_ASTEROIDS)
		gs->asteroids[ gs->numAsteroids++ ] = a;
}
void AddExplosion( GameState *gs, Coord pos )
{
	// Make an explosion
	Explosion e;
	e.pos = pos;
	e.age = 0.0;

	// Add to the end of the list
	if (gs->numExplosions < MAX_EXPLOSIONS)
		gs->explosions[ gs->numExplosions++ ] = e;
}
void AddBlast( GameState *gs, int player, Coord pos, Coord speed )
{
	// Make an blast
	Blast b;
	b.pos = pos;
	b.speed = speed;
	b.age   = 0.0;

	// Add to the end of the list
	if (gs->players[player].numBlasts < MAX_BLASTS)
		gs->players[player].blasts[ gs->players[player].numBlasts++ ] = b;
}
void AddSound( GameState *gs, int type )
{
	// Trigger a sound
	Sound s;
	s.type = type;

	// Add to the end of the list
	if (gs->numSounds < MAX_SOUNDS)
		gs->sounds[ gs->numSounds++ ] = s;
}


//------------------------------------
// Delete the game objects
//   (Copy + Paste anyone?)
//------------------------------------
Asteroid DeleteAsteroid(GameState *gs, int idx) {
	int a;
	Asteroid killed = gs->asteroids[idx];

	// move everything down to replace
	for (a=idx; a<gs->numAsteroids-1; a++)
		gs->asteroids[a] = gs->asteroids[a+1];
	gs->numAsteroids--;  // it's one smaller

	// return the killed asteroid (so we can look at it)
	return killed;
}
Explosion DeleteExplosion(GameState *gs, int idx) {
	int e;
	Explosion killed = gs->explosions[idx];

	// move everything down to replace
	for (e=idx; e<gs->numExplosions-1; e++)
		gs->explosions[e] = gs->explosions[e+1];
	gs->numExplosions--;  // it's one smaller

	// return the killed asteroid (so we can look at it)
	return killed;

}
Blast DeleteBlast(GameState *gs, int player, int idx) {
	int b;
	Blast killed = gs->players[player].blasts[idx];

	// move everything down to replace
	for (b=idx; b<gs->players[player].numBlasts-1; b++)
		gs->players[player].blasts[b] = gs->players[player].blasts[b+1];
	gs->players[player].numBlasts--;  // it's one smaller

	// return the killed asteroid (so we can look at it)
	return killed;

}

float RandF() {
	return ( (float)rand() / (float)RAND_MAX );
}
Coord RandomDirection(float speed) {
	Coord dir;
	float theta = 2.0*PI* RandF();
	dir.x = speed * cos(theta);
	dir.y = speed * sin(theta);
	return dir;
}
Coord RandomPosition(float minX, float maxX, float minY, float maxY)
{
	Coord pos;
	pos.x = minX + (maxX-minX)*RandF();
	pos.y = minY + (maxY-minY)*RandF();
	return pos;
}

